<?php
/**
 * Ninja Saga AMF Gateway - Use SabreAMF_Server for Response
 * 
 * Use Server class to handle response correctly
 */

error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

$baseDir = __DIR__;
$vendorPath = $baseDir . DIRECTORY_SEPARATOR . 'vendor' . DIRECTORY_SEPARATOR . 'autoload.php';

error_log("=== AMF Gateway Start ===");

if (!file_exists($vendorPath)) {
    header('Content-Type: application/json');
    http_response_code(500);
    echo json_encode(['status' => '0', 'error' => 'SabreAMF not installed']);
    exit;
}

require_once $vendorPath;

if (!class_exists('SabreAMF_Server')) {
    header('Content-Type: application/json');
    http_response_code(500);
    echo json_encode(['status' => '0', 'error' => 'SabreAMF classes not found']);
    exit;
}

require_once 'config.php';
require_once 'db.php';

header('Content-Type: application/x-amf');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

require_once 'services/SystemService.php';
require_once 'services/CharacterDAO.php';
require_once 'services/SystemData.php';

class AMFHandler {
    private $db;
    
    public function __construct($db) {
        $this->db = $db;
    }
    
    public function handle($service, $method, $params) {
        $services = array(
            'SystemService' => 'SystemService',
            'CharacterDAO' => 'CharacterDAO',
            'SystemData' => 'SystemData',
        );
        
        if (!isset($services[$service])) {
            throw new Exception("Service not found: $service");
        }
        
        $className = $services[$service];
        $serviceObj = new $className($this->db);
        
        if (!method_exists($serviceObj, $method)) {
            throw new Exception("Method not found: $service.$method");
        }
        
        return call_user_func_array(array($serviceObj, $method), $params);
    }
}

try {
    $db = getDBConnection();
    $handler = new AMFHandler($db);
    
    $input = file_get_contents('php://input');
    error_log("Input length: " . strlen($input) . " bytes");
    
    // Detect AMF version from input (first 2 bytes: 0x0003 = AMF3, 0x0000 = AMF0)
    $version = 0;
    if (strlen($input) >= 2) {
        $version = ord($input[0]) * 256 + ord($input[1]);
        error_log("AMF Version detected: $version (0x" . dechex($version) . ")");
    }
    
    // Use SabreAMF_Server to handle request/response correctly
    // Server.php now stores version in __construct() and uses it in sendResponse()
    $server = new SabreAMF_Server();
    
    // Get requests
    $requests = $server->getRequests();
    error_log("Number of requests: " . count($requests));
    
    if (count($requests) == 0) {
        throw new Exception("No requests in AMF message");
    }
    
    // Process each request
    foreach ($requests as $request) {
        $target = $request['target'];
        $responseUri = $request['response'];
        $data = $request['data'];
        
        error_log("Request: target=$target, response=$responseUri");
        error_log("Request data type: " . gettype($data));
        
        // If data is AMF3_Wrapper, extract it
        if (is_object($data) && $data instanceof SabreAMF_AMF3_Wrapper) {
            $data = $data->getData();
            error_log("AMF3 data unwrapped");
        } elseif (is_array($data) && isset($data[0]) && is_object($data[0]) && $data[0] instanceof SabreAMF_AMF3_Wrapper) {
            $data = $data[0]->getData();
            error_log("AMF3 data unwrapped from array");
        }
        
        // Parse target
        if (strpos($target, '.') !== false) {
            list($service, $method) = explode('.', $target, 2);
        } else {
            $service = $target;
            $method = 'login';
        }
        
        // Get params
        if (is_array($data)) {
            $params = $data;
        } elseif ($data !== null) {
            $params = array($data);
        } else {
            $params = array('test', 'test123', BUILD_NO);
        }
        
        error_log("Calling: $service::$method with " . count($params) . " params");
        
        // Handle request
        try {
            $result = $handler->handle($service, $method, $params);
            
            // Format response
            $responseObj = new stdClass();
            $responseObj->status = '1';
            $responseObj->result = $result;
            $responseObj->signature = md5(json_encode($result) . APP_SECRET);
            $responseObj->account_lock = null;
            $responseObj->country_area = null;
            $responseObj->isNewAccount = null;
            $responseObj->promote_id = null;
            $responseObj->isTrialEmblem = 0;
            $responseObj->isExpired = 0;
            
            // Set response using Server class
            $server->setResponse(
                $responseUri,
                SabreAMF_Const::R_RESULT,
                $responseObj
            );
            
            error_log("Response set for: $responseUri");
            
        } catch (Exception $e) {
            error_log("Error handling request: " . $e->getMessage());
            $errorResponse = new stdClass();
            $errorResponse->status = '0';
            $errorResponse->error = $e->getMessage();
            
            // Set error response
            $server->setResponse(
                $responseUri,
                SabreAMF_Const::R_STATUS,
                $errorResponse
            );
        }
    }
    
    // Send response using Server class
    // This handles the format correctly
    error_log("Sending response...");
    $server->sendResponse();
    error_log("Response sent successfully");
    
} catch (Exception $e) {
    error_log("Error: " . $e->getMessage());
    error_log("Stack: " . $e->getTraceAsString());
    
    header('Content-Type: application/json');
    echo json_encode(array(
        'status' => '0',
        'error' => $e->getMessage()
    ));
}
?>
